import { useToast } from '@chakra-ui/react';
import apiFetch from '@wordpress/api-fetch';
import {
	Button,
	Modal,
	RadioControl,
	TextControl,
	ToggleControl,
	Tooltip,
} from '@wordpress/components';
import { __ } from '@wordpress/i18n';
import React from 'react';
import { Controller, FormProvider, useForm } from 'react-hook-form';
import { useMutation } from 'react-query';
import { z } from 'zod';
import { localized } from '../../utils/global';

const formSchema = z.object({
	license_key: z.string().min(32).max(32),
	is_diagnostic_tracking_allowed: z
		.enum(['true', 'false'])
		.default('true')
		.optional(),
	is_extensions_tracking_allowed: z
		.enum(['true', 'false'])
		.default('true')
		.optional(),
	is_marketing_allowed: z.enum(['true', 'false']).default('false').optional(),
	email: z.string().email(),
});

type FormSchema = {
	license_key: string;
	is_diagnostic_tracking_allowed: 'true' | 'false';
	is_extensions_tracking_allowed: 'true' | 'false';
	is_marketing_allowed: 'true' | 'false';
	email: string;
};

export const ActivationForm = () => {
	const [isDetailsOpen, setIsDetailsOpen] = React.useState(false);
	const [isEmailModalOpen, setIsEmailModalOpen] = React.useState(false);

	const toast = useToast();

	const form = useForm<FormSchema>({
		defaultValues: {
			license_key: '',
			is_diagnostic_tracking_allowed: 'true',
			is_extensions_tracking_allowed: 'true',
			is_marketing_allowed: 'true',
			email: '',
		},
		mode: 'all',
	});

	const activationMutation = useMutation(
		(data: FormSchema) => {
			const formData = new FormData();
			for (let x in data) {
				formData.append(x, data[x as keyof FormSchema]);
			}
			formData.append('module_id', localized.fs?.module_id);
			formData.append('action', localized.fs?.activation?.action);
			formData.append('security', localized.fs?.activation?.security);
			return new Promise((resolve, reject) => {
				apiFetch({
					url: localized.fs?.ajax_url,
					method: 'POST',
					body: formData,
				})
					.then((res: any) => {
						if (!res.success) {
							reject(new Error(res.error));
						} else {
							resolve(res);
						}
					})
					.catch((err) => {
						reject(err);
					});
			});
		},
		{
			onSuccess() {
				toast({
					status: 'success',
					isClosable: true,
					title: __('License activated successfully!', 'colormag'),
				});
				window.location.reload();
			},
			onError(e: Error) {
				toast({
					status: 'error',
					isClosable: true,
					title: e.message,
				});
			},
		},
	);

	const handleActivation = (data: FormSchema) => {
		activationMutation.mutate(data);
	};

	return (
		<FormProvider {...form}>
			<form
				onSubmit={form.handleSubmit(handleActivation)}
				className="cm-freemius-form max-w-2xl mx-auto"
			>
				<div className="bg-[#FFFFFF] border border-[#F4F4F4] p-6 rounded-lg">
					<div className="mb-6">
						<h2 className="text-2xl font-semibold mb-2">
							{__('Activate License', 'colormag')}
						</h2>
						<p className="text-gray-700">
							{__(
								'Please enter the license key that you received in the email right after the purchase!',
								'colormag',
							)}
						</p>
					</div>

					<div className="mb-4">
						<Controller
							name="license_key"
							control={form.control}
							rules={{
								required: true,
								validate: (value) => {
									return value.length === 32;
								},
							}}
							defaultValue=""
							render={({ field }) => (
								<div>
									<label htmlFor="license_key"></label>
									<input
										type="password"
										{...field}
										id="license_key"
										className="w-full"
									/>
								</div>
							)}
						/>
						{form.formState.errors.license_key && (
							<p className="text-red-500 text-sm">
								{form.formState.errors.license_key.message}
							</p>
						)}
					</div>

					<div className="flex gap-4 items-center mb-6">
						<Button
							type="submit"
							variant="primary"
							disabled={activationMutation.isLoading || !form.formState.isValid}
							className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed"
						>
							{activationMutation.isLoading ? (
								<div className="flex items-center gap-2">
									<svg
										className="animate-spin h-4 w-4 text-white"
										xmlns="http://www.w3.org/2000/svg"
										fill="none"
										viewBox="0 0 24 24"
									>
										<circle
											className="opacity-25"
											cx="12"
											cy="12"
											r="10"
											stroke="currentColor"
											strokeWidth="4"
										></circle>
										<path
											className="opacity-75"
											fill="currentColor"
											d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
										></path>
									</svg>
									{__('Activating...', 'colormag')}
								</div>
							) : (
								__('Activate', 'colormag')
							)}
						</Button>
						<Button
							variant="secondary"
							onClick={() => setIsEmailModalOpen(true)}
							disabled={activationMutation.isLoading}
							className="text-blue-600 hover:text-blue-700 disabled:opacity-50 disabled:cursor-not-allowed"
						>
							{__("Can't find your License Key?", 'colormag')}
						</Button>
					</div>
					{form.formState.isValid && (
						<div className="border border-blue-600 bg-blue-50 rounded p-5 mb-6">
							<p className="font-semibold mb-4">
								{__(
									"Please let us know if you'd like us to contact you for security & feature updates, educational content, and occasional offers:",
									'colormag',
								)}
							</p>
							<Controller
								name="is_marketing_allowed"
								control={form.control}
								render={({ field }) => (
									<RadioControl
										selected={field.value}
										options={[
											{
												label: __(
													'Yes - send me security & feature updates, educational content and offers.',
													'colormag',
												),
												value: 'true',
											},
											{
												label: __(
													'No - do NOT send me security & feature updates, educational content and offers.',
													'colormag',
												),
												value: 'false',
											},
										]}
										onChange={field.onChange}
									/>
								)}
							/>
						</div>
					)}

					<div className="border border-gray-200 rounded-lg p-4">
						<div className="flex items-center justify-between mb-4">
							<p className="m-0">
								{__(
									'For delivery of security & feature updates, and license management, ColorMag Pro needs to',
									'colormag',
								)}
							</p>
							<Button
								onClick={() => setIsDetailsOpen(!isDetailsOpen)}
								className="p-1 border border-gray-300 rounded"
							>
								<svg
									className={`w-4 h-4 transform transition-transform ${isDetailsOpen ? 'rotate-180' : ''}`}
									xmlns="http://www.w3.org/2000/svg"
									viewBox="0 0 20 20"
									fill="currentColor"
								>
									<path
										fillRule="evenodd"
										d="M5.293 7.293a1 1 0 011.414 0L10 10.586l3.293-3.293a1 1 0 111.414 1.414l-4 4a1 1 0 01-1.414 0l-4-4a1 1 0 010-1.414z"
										clipRule="evenodd"
									/>
								</svg>
							</Button>
						</div>

						{isDetailsOpen && (
							<div className="space-y-4">
								<div className="border-t border-gray-200 pt-4">
									<Tooltip
										text={__(
											'To let you manage & control where the license is activated',
											'colormag',
										)}
									>
										<h4 className="font-semibold mb-2">
											{__('View License Essential', 'colormag')}
										</h4>
									</Tooltip>
									<p className="text-gray-600">
										{__(
											'Homepage URL, Plugin version, SDK version',
											'colormag',
										)}
									</p>
								</div>

								<div className="flex items-center justify-between">
									<div>
										<Tooltip
											text={__(
												'To ensure compatibility with your site',
												'colormag',
											)}
										>
											<h4 className="font-semibold mb-2">
												{__('View Diagnostic Info (optional)', 'colormag')}
											</h4>
										</Tooltip>
										<p className="text-gray-600">
											{__(
												'WordPress & PHP versions, site language & title',
												'colormag',
											)}
										</p>
									</div>
									<Controller
										name="is_diagnostic_tracking_allowed"
										control={form.control}
										render={({ field }) => (
											<ToggleControl
												__nextHasNoMarginBottom
												label=""
												checked={field.value === 'false'}
												onChange={(value) =>
													field.onChange(value ? 'false' : 'true')
												}
											/>
										)}
									/>
								</div>

								<div className="flex items-center justify-between">
									<div>
										<Tooltip
											text={__(
												'To ensure compatibility and avoid conflicts',
												'colormag',
											)}
										>
											<h4 className="font-semibold mb-2">
												{__(
													'View Plugins & Themes List (optional)',
													'colormag',
												)}
											</h4>
										</Tooltip>
										<p className="text-gray-600">
											{__(
												'Names, slugs, versions, and if active or not',
												'colormag',
											)}
										</p>
									</div>
									<Controller
										name="is_extensions_tracking_allowed"
										control={form.control}
										render={({ field }) => (
											<ToggleControl
												__nextHasNoMarginBottom
												label=""
												checked={field.value === 'true'}
												onChange={(value) =>
													field.onChange(value ? 'true' : 'false')
												}
											/>
										)}
									/>
								</div>
							</div>
						)}
					</div>
				</div>

				{isEmailModalOpen && (
					<Modal
						title={__('Send License Key', 'colormag')}
						onRequestClose={() => setIsEmailModalOpen(false)}
						className="wp-core-ui"
					>
						<div className="p-6">
							<p className="mb-4">
								{__(
									"Enter the email address you've used during the purchase and we will resend you the license key.",
									'colormag',
								)}
							</p>
							<Controller
								name="email"
								control={form.control}
								render={({ field }) => (
									<TextControl
										label={__('Email Address', 'colormag')}
										value={field.value}
										onChange={field.onChange}
										type="email"
									/>
								)}
							/>
							<Button
								variant="primary"
								className="w-full mt-4 text-center bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700"
								onClick={() => {
									setIsEmailModalOpen(false);
								}}
							>
								{__('Send License Key', 'colormag')}
							</Button>
						</div>
					</Modal>
				)}
			</form>
		</FormProvider>
	);
};
